/*
 * MkScript 1.1 by Duane McDonnell <dmcdon@bigfoot.com>.
 * Incorporates code from txtelite by Ian Bell <iancgbell@email.com>.
 *
 * MkScript generates scripts for txtelite version 1.3 or higher by
 * Ian Bell. Written for the True Millennium Challenge, the program
 * attempts to reach a given number of credits in the minimum number
 * of jumps (in the case of the Challenge, the credits was set to
 * 1,000,000).
 *
 * This is a fairly naive implementation which could certainly be
 * improved upon in terms of reducing the number of jumps taken
 * as well as the running time and memory requirements.
 *
 * The following setting was used for the Challenge:
 *
 * mkscript 1000000 7 7 sinclair.txt  (632 jumps)
 *
 * Permission is given to freely distribute and modify this code.
 * If you make any improvements to the algorithm, please advise
 * the author. Some suggestions for improvement in various areas
 * are given here.
 *
 * Credit/Jump Improvements:
 * - The current algorithm always buys at one system and sells at the
 *   next. Sometimes you can make more profit by buying some goods at
 *   one system and selling them 'further down the line' (not necessarily
 *   the next system). Hint: you'll need to modify the State structure
 *   so that it can track goods for buying and selling separately
 *   (at the moment it simply tracks which goods we're carrying).
 *
 * - Currently the paths which can be followed from a planet are followed
 *   in the order in which they are determined by the distance function
 *   (which determines in-range systems). This is not the best criteria
 *   for selecting paths to follow when the [paths] argument is set lower
 *   than the number of paths which can be followed from the system. Some
 *   form of path weighting should really be implemented so that more
 *   profitable paths are taken first (this could be done by looking
 *   ahead and sorting paths into most=>least profitable order and
 *   following the more profitable paths first). If [paths] is set
 *   sufficiently high, this doesn't cause any problems since all
 *   paths will be followed.
 *
 * Speed Improvements:
 *
 * - Cache markets in a hash table (use the market seed as the hash key).
 *
 * - Use squared distances in get_distance() when determining in-range
 *   systems.
 *
 * - Better lookahead analysis of paths and pruning.
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/****************** IB's types/structs/defines follow *********************/

typedef int boolean;
typedef unsigned char uint8;
typedef unsigned short uint16;
typedef signed short int16;
typedef signed long int32;
typedef uint16 uint;
typedef int planetnum;
#define true (-1)
#define false (0)
#define tonnes (0)

typedef struct
{ uint8 a,b,c,d;
} fastseedtype;  /* four byte random number used for planet description */

typedef struct
{ uint16 w0;
  uint16 w1;
  uint16 w2;
} seedtype;  /* six byte random number used as seed for planets */

typedef struct
{   uint x;
   uint y;       /* One byte unsigned */
   uint economy; /* These two are actually only 0-7  */
   uint govtype;   
   uint techlev; /* 0-16 i think */
   uint population;   /* One byte */
   uint productivity; /* Two byte */
   uint radius; /* Two byte (not used by game at all) */
   fastseedtype   goatsoupseed;
   char name[12];
} plansys ;

#define galsize (256)
#define AlienItems (16)
#define lasttrade AlienItems

#define numforLave 7       /* Lave is 7th generated planet in galaxy one */
#define numforZaonce 129
#define numforDiso 147
#define numforRied 46

seedtype seed;

fastseedtype rnd_seed;

typedef struct
{                         /* In 6502 version these were: */
   uint baseprice;        /* one byte */
   int16 gradient;        /* five bits plus sign */
   uint basequant;        /* one byte */
   uint maskbyte;         /* one byte */
   uint units;            /* two bits */
   char name[20];         /* longest="Radioactives" */
  } tradegood ;

typedef struct
{  uint quantity[lasttrade+1];
  uint price[lasttrade+1];
} markettype ;

int fuelcost =2; /* 0.2 CR/Light year */
int maxfuel =70; /* 7.0 LY tank */

const uint16 base0=0x5A4A;
const uint16 base1=0x0248;
const uint16 base2=0xB753;  /* Base seed for galaxy 1 */

char pairs0[]="ABOUSEITILETSTONLONUTHNO";
/* must continue into .. */
char pairs[] = "..LEXEGEZACEBISO"
               "USESARMAINDIREA."
               "ERATENBERALAVETI"
               "EDORQUANTEISRION"; /* Dots should be nullprint characters */

char unitnames[][5] ={"t","kg","g"};

/* Data for DB's price/availability generation system */
/*                   Base  Grad Base Mask Un   Name
                     price ient quant     it              */ 

#define POLITICALLY_CORRECT   0
/* Set to 1 for NES-sanitised trade goods */

tradegood commodities[]=
                   {
                    {0x13,-0x02,0x06,0x01,0,"Food        "},
                    {0x14,-0x01,0x0A,0x03,0,"Textiles    "},
                    {0x41,-0x03,0x02,0x07,0,"Radioactives"},
#if POLITICALLY_CORRECT
                    {0x28,-0x05,0xE2,0x1F,0,"Robot Slaves"},
                    {0x53,-0x05,0xFB,0x0F,0,"Beverages   "},
#else
                    {0x28,-0x05,0xE2,0x1F,0,"Slaves      "},
                    {0x53,-0x05,0xFB,0x0F,0,"Liquor/Wines"},
#endif 
                    {0xC4,+0x08,0x36,0x03,0,"Luxuries    "},
#if POLITICALLY_CORRECT
                    {0xEB,+0x1D,0x08,0x78,0,"Rare Species"},
#else
                    {0xEB,+0x1D,0x08,0x78,0,"Narcotics   "},
#endif 
                    {0x9A,+0x0E,0x38,0x03,0,"Computers   "},
                    {0x75,+0x06,0x28,0x07,0,"Machinery   "},
                    {0x4E,+0x01,0x11,0x1F,0,"Alloys      "},
                    {0x7C,+0x0d,0x1D,0x07,0,"Firearms    "},
                    {0xB0,-0x09,0xDC,0x3F,0,"Furs        "},
                    {0x20,-0x01,0x35,0x03,0,"Minerals    "},
                    {0x61,-0x01,0x42,0x07,1,"Gold        "},
                    {0xAB,-0x02,0x37,0x1F,1,"Platinum    "},
                    {0x2D,-0x01,0xFA,0x0F,2,"Gem-Strones "},
                    {0x35,+0x0F,0xC0,0x07,0,"Alien Items "},
                   };

plansys makesystem(seedtype *s);
void nextgalaxy(seedtype *s);
markettype genmarket(uint fluct, plansys p);
signed int ftoi(double value);

/*********************** MKSCRIPT STUFF STARTS HERE ***********************/
//#define NULL ((void *)0)
typedef char byte;
typedef unsigned char ubyte;
typedef short word;
typedef unsigned short uword;
typedef unsigned long ulong;

/* Prototypes */
void port_srand(unsigned int);
int port_rand(void);
boolean get_segment(ulong depth);
void free_segment(void);
void process_segment(struct Node *term, boolean end);
boolean attach_nodes(void);
void free_nodes(void);
void init_node(struct Node *curr);
void reset_node(struct Node *node);
void scan_tree(struct Node *node);
void reset_tree(void);
void buy_stuff(struct Node *curr);
uint get_market_fluctuation(int seed, int *newseed);
ulong get_distance(plansys *a,plansys *b);
void create_universe(void);

/* An array of Segment structures is used to sort routes from tree
 * segment traversals into first => last order since a tree traversal
 * returns a child node (the node's parent field is used to recreate
 * the correct jump order).
 */
struct Segment
{
   struct   Node *node;
};

/* State buffers store the data defining a game state during transit
 * from one system to another. Specifically, it stores which trade
 * goods are in the hold, what the size of the hold is, the fuel used
 * for the jump (and hence the fuel remaining since we always fill
 * to maximum prior to jumping) and any cash representing savings.
 *
 * There is no state buffer when you first start at Lave (the node
 * is sufficient to store the state) but all other systems reached
 * via jumping have an associated state buffer (including Lave if
 * we happen to revisit it).
 */
struct State
{
   /* Left-over cash after buying goods or ship ugrades. This gets
    * passed on to our child node (planet/system) upon arrival.
    */
   ulong    cash;
   /* Stores the amount of fuel used to make the jump. This allows us
    * to minimally fill the tank when we arrive. Bit 7 is also set
    * to flag that we upgraded the cargo bay to 35t capacity. This is
    * because our child node will need to have its NODEFLAG_LARGEBAY
    * flag set upon arrival.
    */
   ubyte    fuel;
   /* This stores the amount of each good which we're carrying using
    * the same LUT indexing which txtelite uses.
    */
   ubyte    goods[lasttrade+1];           /* [0...lasttrade] */
};

/* Stores the possible paths which can be followed from a system.
 */
struct Path
{
   struct   Node *path;
   struct   State state;
};

/* Each tree node can be thought of as being a planet/system.
 *
 * pathnum tells us the path number (offset in parent's paths table from
 * 0 onwards) which was used to get from the parent to this node/planet.
 * This is required because we need to be able to access the correct
 * state buffer for the path which was followed since this will differ
 * for each of the paths. If parent is NULL, pathnum is invalid.
 *
 * level starts from 1 if we are the root node (which also implies
 * that parent will be NULL) and increments as we move down tree levels.
 * If level is the same as the tree depth then we are a leaf node (we
 * have no children). This condition is used to terminate recursion.
 *
 * pnum is the planet number we're in. This ranges from 0 to 2047
 * (8 galaxies * 256 planets) and therefore also encodes the galaxy
 * which we're in (0 - 7 after >> 8).
 *
 * visitcount ranges from 0 onwards and stores the current offset into
 * the paths array for the last path followed from here. This count
 * is incremented when it comes time to follow a new path. When the
 * count exceeds the maximum number of paths which can be followed
 * we commence backtracking.
 */
struct Node
{
   struct   Node *parent;     /* NULL = this is the root node */
   long     cash;             /* Cash after selling goods/upgrading ship */
   int      mktseed;          /* Market seed; platform independent */
   uword    pnum;             /* Planet number (0 - 2047; encodes galaxy) */
   ubyte    level;            /* Node's level in the tree (1 = root) */
   ubyte    pathnum;          /* Parent's path number used to get here */
   ubyte    visitcount;       /* Last in-range planet jumped to from here */
   ubyte    flags;            /* See NODEFLAG_xxx definitions */
   struct   Path paths[1];    /* Paths we can follow from here */
};

/* Node flag definitions */
#define NODEFLAG_LAVESTART    (1<<0)   /* Started from Lave station */
#define NODEFLAG_LARGEBAY     (1<<1)   /* Have a 35t capacity cargo bay */

/* Global tree structure.
 *
 * Depth = 0 is an empty tree, depth = 1 is a tree with only a root
 * node, etc.
 */
struct Tree
{
   /* These fields should be filled out before attaching nodes to
    * the tree.
    */
   struct   Node *root;       /* Pointer to root node or NULL if empty */
   ulong    depth;            /* Depth of tree (number of levels) */
   ulong    maxnodepaths;     /* # of paths we can follow at each node */

   ulong    totnodes;         /* Total number of nodes in the tree */
   ulong    nodesize;         /* Size of a node; not sizeof(struct Node)! */
   ulong    totmem;           /* Total memory reserved for holding nodes */
} Tree;

#define BUYDEBUG                 0     /* Set for buy debugging info */

#define STARTCREDITS       (100*10)    /* Lave starting credits */

#define LARGEBAYTHRESHOLD  (500*10)    /* Min. cash before buying 35t bay */
#define LARGEBAYCOST       (400*10)

#define INTERGALTHRESHOLD  (100000*10) /* Min. cash before buying IG drive */
#define INTERGALCOST       (5000*10)

/* Startup related */
long     Depth;               /* [depth]      */
long     Jumps;               /* [paths]      */
long     Credits;             /* [credits]    */
FILE     *fh = NULL;          /* [scriptfile] */


struct   Node *NextNode;   /* Next free node (for tree initialization) */

struct   Node *BestSegmentNode;        /* Best partial segment node */
struct   Node *BestTerminatedNode;     /* Best terminated route node */

struct   Segment *SegArray;

struct   State *StateBuffer;

ulong    MaxCash;
ulong    TotJumps;

ulong    PlanetStart;
ulong    PlanetStop;
ulong    CurrentPlanet;
ulong    FindCount;
ulong    Distance;
ulong    Count;
struct   Node *JumpNode;

ulong    StopLevel = 0x7fffffff;
boolean  HaveBigBay = false;

ulong    SortArray[lasttrade+1+1];

markettype ParentMarket;
markettype LocalMarket;

/* Precalculated storage for Elite universe (8 galaxies).
 *
 * Planets in galaxy #1 are indexed using [0...255]
 * Planets in galaxy #2 are indexed using [256...511]
 * ...
 * ...
 * Planets in galaxy #8 are indexed using [1792...2047]
 */
plansys  Universe[galsize * 8];


static unsigned int lastrand = 0;

void port_srand(unsigned int seed)
{
    lastrand = seed - 1;
}

int port_rand(void)
{
    int t;

    t = (((((((((((lastrand << 3) - lastrand) << 3)
        + lastrand) << 1) + lastrand) << 4)
        - lastrand) << 1) - lastrand) + 0xe60)
        & 0x7fffffff;

    lastrand = t - 1;   
    return t;
}

int main(int argc, char **argv)
{
   boolean  showusage = true;

   printf("MkScript 1.1 by Duane McDonnell <dmcdon@bigfoot.com>\n");
   printf("Incorporates code by Ian Bell <iancgbell@email.com>\n\n");

   if (argc == 5)
   {
      Credits = atof(argv[1]) * 10.0;

      if (Credits > 1000)
      {
         if ((Depth = atoi(argv[2])) > 1)
         {
            if ((Jumps = atoi(argv[3])) > 0)
            {
               if (fh = fopen(argv[4], "wb"))
                  showusage = false;

               else
                  printf("Error: Can't open \"%s\" for output!\n\n", argv[4]);
            }

            else
               printf("Error: [paths] must be > 0!\n\n");
         }

         else
            printf("Error: [depth] must be > 1!\n\n");
      }

      else
         printf("Error: [credits] must be > 100!\n\n");
   }

   if (showusage)
   {
      printf("Usage: %s [credits] [depth] [paths] [scriptfile]\n\n", argv[0]);

      printf("[credits]     The number of credits you wish to make. Must be > 100.\n\n");

      printf("[depth]       The depth of the segment tree to generate. The higher this\n");
      printf("              value the longer it will take to generate a script but the\n");
      printf("              more optimal the script should be. Must be > 1 (but set as\n");
      printf("              high as you have memory for and can bear to wait).\n\n");

      printf("[paths]       The maximum number of jump paths to consider at each planet.\n");
      printf("              This value must be > 0. A value of 1 will check one path, 2\n");
      printf("              will check 2 paths, and so on. The recommended setting is 7\n");
      printf("              to check all paths including intergalactic jumps.\n\n");

      printf("[scriptfile]  A file to save the generated script file to. You can then\n");
      printf("              redirect it into txtelite.exe (txtelite.exe < [scriptfile]).\n");

      return 0;
   }

   /* Initialize script with a 'rand' command since we want
    * to use the platform independent RNG (requires txtelite
    * version 1.3 or higher).
    */
   fprintf(fh, "rand\n");

   /* Allocate segment array */
   
   if (!(get_segment(Depth)))
      return 0;

   /* Initialize global tree */

   printf("Initializing...\n");

   Tree.depth = Depth;
   Tree.maxnodepaths = Jumps;
   Tree.root = NULL;

   /* Attach nodes to tree */

   if (!(attach_nodes()))
      return 0;

   /* Create Elite universe (8 * 256 planets) */

   create_universe();
      
   /* We start off at the root node which represents Lave. Then we
    * enter a segment processing loop, traversing the tree until
    * it returns a child node representing a profitable series
    * of jumps. After the segment has been processed, the tree
    * is reset for the next segment, with the new root node being
    * the end node of the previous scan. The loop continues in
    * this manner until we reach our terminating credits.
    *
    * Note that Lave is a special case as far as market fluctuations
    * go. The fluctuation for Lave is not generated from the market
    * seed but is hardwired to 0x00. All other systems' markets are
    * generated from their market seeds. Therefore, Lave needs to
    * be flagged as a special case so that the tree traverser can
    * reconstruct Lave's market prices. This is done by setting
    * the NODEFLAG_LAVESTART bit in the node's flags field.
    *
    * We use the portable version of txtelite's srand() and rand()
    * so a script 'rand' command must be output first. Using the
    * portable version also means that we have access to the
    * internal seed so we don't need to worry about tracking
    * ANSI rand() calling counters (you can't peek the internal
    * seed with the ANSI functions and the value passed to
    * srand() may or may not be the internally maintained seed).
    */
   port_srand(12345);                  /* Ensure repeatability */
   Tree.root->mktseed = lastrand;      /* We use the internal seed! */
   Tree.root->cash = STARTCREDITS;     /* Starting cash = 100CR */
   Tree.root->pnum = numforLave;       /* Planet number (galaxy #1) */
   Tree.root->flags |= NODEFLAG_LAVESTART;

   /* Segment processing loop */

   for (;;)
   {
      /* Get next segment */

      MaxCash = 0;
      scan_tree(Tree.root);

      /* Process segment */

      if (BestTerminatedNode)
      {
         /* This is the final segment */
         
         process_segment(BestTerminatedNode, true);
         fprintf(fh, "q\n");
         break;
      }

      /* This is an intermediate segment (more to come) */

      process_segment(BestSegmentNode, false);

      /* Initialize root node for next segment */

      reset_node(Tree.root);

      /* Inherit settings for next segment (segment wrap-around).
       * We can no longer be at Lave prior to making our first
       * jump so the LAVESTART flag also needs to be cleared.
       */
      Tree.root->mktseed = BestSegmentNode->mktseed;
      Tree.root->cash = BestSegmentNode->cash;
      Tree.root->pnum = BestSegmentNode->pnum;
      Tree.root->flags = BestSegmentNode->flags & ~NODEFLAG_LAVESTART;

      /* Now reset the rest of the tree (the root node has
       * already been done; reset_tree() skips over the root).
       */
      reset_tree();
   }

   if (fh)
      fclose(fh);

   free_nodes();
   free_segment();

   return 0;
}

/* Allocate segment array.
 *
 * Returns true if okay, else not enough memory.
 */
boolean get_segment(ulong depth)
{
   /* + 1 for terminator */
   if (!(SegArray = calloc(1, sizeof(struct Segment) * (depth + 1))))
   {
      printf("Not enough memory!\n");
      return false;
   }
      
   return true;
}

/* Free segment array returned by get_segment().
 * NULL is also safe to pass.
 */
void free_segment(void)
{
   if (SegArray)
   {
      free(SegArray);
      SegArray = NULL;
   }
}

/* Fill in the Tree structure and call here to attach nodes to it.
 * Returns true if successful, else not enough memory available etc.
 */
boolean attach_nodes(void)
{
   ulong    level = 1;
   ulong    t = 1;
   boolean  overflow = false;

   ulong    t_last;
   ulong    tot_last;

   if (Tree.depth <= 1 || Tree.maxnodepaths == 0)
   {
      printf("Bad [depth] and/or [paths] value given\n");
      return false;
   }

   Tree.totnodes = 1;            /* Special case: count top node */

   t_last = t;
   tot_last = Tree.totnodes;

   /* Calculate how many nodes the tree will have given the
    * maximum number of paths possible from each node and
    * the tree's depth. We also check for overflows here
    * as these can easily occur.
    */
   while (level++ < Tree.depth)
   {
      t *= Tree.maxnodepaths;

      if (t_last > t)
         overflow = true;        

      t_last = t;

      Tree.totnodes += t;

      if (tot_last >= Tree.totnodes)
         overflow = true;

      tot_last = Tree.totnodes;
   }

   Tree.nodesize = sizeof(struct Node) + (Tree.maxnodepaths * sizeof(struct Path));

   if (Tree.nodesize >= 0x80000000)
      overflow = true;  
   
   Tree.totmem = Tree.totnodes * Tree.nodesize;

   if ((Tree.totmem >= 0x80000000) || (Tree.nodesize >= Tree.totmem))
      overflow = true;

   if (overflow)
   {
      printf("Error: Overflow occurred due to too many nodes in tree.\n");
      printf("       Try using a lower [depth] or [paths] setting.\n");
      return false;
   }

   /* Allocate memory for all nodes. The nodes are stored
    * in a contiguous array since the tree won't need to
    * be resized and we won't be doing any insertions or
    * deletions on it.
    */
   if (!(Tree.root = calloc(1, Tree.totmem)))
   {
      printf("Not enough memory for tree. Need %ld bytes (%.3fMB)\n", Tree.totmem, (float)Tree.totmem / 1048576);
      return false;
   }

   /* Recursively initialize all nodes. The root node
    * is handled as a special case.
    */
   Tree.root->parent = NULL;
   Tree.root->level = 1;
   NextNode = Tree.root;      /* Allocate next node AFTER this one! */

   init_node(Tree.root);      /* Traverse tree, initializing nodes... */

   printf("Attached %ld nodes using %ld bytes (%.3fMB)\n", Tree.totnodes, Tree.totmem, (float)Tree.totmem / 1048576);
   return true;
}

/* Free memory allocated by attach_nodes().
 * NULL is also safe to pass.
 */
void free_nodes(void)
{
   if (Tree.root)
   {
      free(Tree.root);
      Tree.root = NULL;
   }
}

/* Initialize a node and recursively initialize its child nodes.
 */
void init_node(struct Node *curr)
{
   ulong c;

   /* Calculate the level of this node */

   if (curr->parent)
      curr->level = curr->parent->level + 1;    /* One level down */

   /* Stop recursing if this is a leaf node */

   if (curr->level == Tree.depth)
      return;

   /* Now initialize all of the child nodes of this node. This
    * node becomes the parent of these children. Each of these
    * child nodes represents a possible planet to be visited.
    */
   for (c = 0; c < Tree.maxnodepaths; c++)
   {
      NextNode = (struct Node *)((ulong)NextNode + Tree.nodesize);
      curr->paths[c].path = NextNode;
      NextNode->parent = curr;
      init_node(NextNode);          /* Descend down this path */
   }
}

/* Do a minimal reset on the tree so that it can be re-used for
 * getting the next segment.
 *
 * Note that the root node is not reset as it's a special case
 * due to the segmented nature of the algorithm. The root node
 * should be reset separately.
 */
void reset_tree(void)
{
   struct   Node *curr = Tree.root;
   long     c = Tree.totnodes;

   while (--c > 0)
   {
      curr = (struct Node *)((ulong)curr + Tree.nodesize);
      reset_node(curr);
   }  
}

/* Reset a single node (no traversal).
 */
void reset_node(struct Node *node)
{
   ulong    i;

   /* Initialize node fields */

   node->cash = 0;
   node->visitcount = 0;
   node->pathnum = 0;
   node->flags = 0;

   /* Clear state buffers */
      
   for (i = 0; i < Tree.maxnodepaths; i++)
      memset(&node->paths[i].state, 0, sizeof(struct State));
}

/* Process tree segment by spitting out txtelite commands to the
 * script file handle.
 *
 * Set 'end' to true if this is the final segment.
 *
 * termnode should point to the terminating Node structure of a
 * segment. That is, it should be the node which the last traversal
 * of the tree returned. If 'end' is true, this node represents
 * the final system we end up at otherwise it's the terminating
 * node of an intermediate segment.
 *
 * Note that the number of credits output to stdout for each jump
 * may not be accurate. It's intended as a gauge only. While the
 * final number of credits for the last jump is always reported
 * correctly (so that it agrees with the generated script), any
 * intermediate jumps may have some of the total cash tied up
 * in the form of fuel and goods in the cargo bay. While this
 * could be corrected, it would involve extra complications due
 * to the segmented nature of the algorithm. Also note that the
 * number of credits output may appear to get smaller between
 * jumps. This is just an artifact and isn't really happening
 * (it just means that more cash is tied up in fuel/goods than
 * for the previous jump).
 */
void process_segment(struct Node *term, boolean end)
{
   char     strbuf[512];
   struct   Segment *currseg;
   struct   State *state;
   struct   Node *curr;
   struct   Node *next;
   char     *str;
   ulong    len;
   ulong    c;
   ubyte    galnum;
   char     oldchar;

   /* Sort nodes into first => last order (we're at 'last'
    * so we follow the parent links to get to 'first').
    */
   currseg = SegArray + Depth;
   currseg->node = NULL;               /* Terminate array */
   curr = term;

   do
   {  
      (--currseg)->node = curr;
      curr = curr->parent;
   } while (curr);

   galnum = term->pnum >> 8;           /* Start galaxy for segment */

   while (curr = currseg++->node)
   {
      str = &strbuf[0];
      *str = '\0';

      if (curr->parent)
      {
         state = &curr->parent->paths[curr->pathnum].state;

         /* Sell goods */

         for (c = 0; c <= lasttrade; c++)
         {
            if (state->goods[c])
            {
               /* Work around "GEM-STRONES" typo in txtelite
                * (fixing txtelite's typo without keeping the
                * typo version as a valid alias would break
                * our scripts, so we truncate to make sure
                * it will always work).
                */
               oldchar = commodities[c].name[4];
               commodities[c].name[4] = '\0';
               
               str += sprintf(str, "s %s %ld\n",
                      commodities[c].name,
                      state->goods[c]);

               commodities[c].name[4] = oldchar;
            }
         }

         /* Update fuel */

         if (end)       /* Don't get fuel after last jump of last segment */
         {
            if (currseg->node)
            {
               if (state->fuel & 0x7f)
                  str += sprintf(str, "f %.1f\n", ((float)(state->fuel & 0x7f)) / 10.0);
            }

            else        /* Make end cash agree with txtelite */
               curr->cash += (state->fuel & 0x7f) * fuelcost;
         }

         else
         {
            if (state->fuel & 0x7f)
               str += sprintf(str, "f %.1f\n", ((float)(state->fuel & 0x7f)) / 10.0);
         }

         /* Output jump/credit progress statistics */

         printf("%ld. %.1f\n", ++TotJumps, ((float)curr->cash) / 10.0);
      }

      /* Cargo hold */

      if (!HaveBigBay)
      {
         if (curr->flags & NODEFLAG_LARGEBAY)
         {
            str += sprintf(str, "cash -400\nhold 35\n");
            HaveBigBay = true;
         }
      }

      if (next = currseg->node)
      {
         state = &curr->paths[next->pathnum].state;

         /* Buy goods */

         for (c = 0; c <= lasttrade; c++)
         {
            if (state->goods[c])
            {
               oldchar = commodities[c].name[4];
               commodities[c].name[4] = '\0';
               
               str += sprintf(str, "b %s %ld\n",
                      commodities[c].name,
                      state->goods[c]);

               commodities[c].name[4] = oldchar;
            }
         }

         /* Jump */

         if ((next->pnum >> 8) != galnum)
         {
            galnum = next->pnum >> 8;
            str += sprintf(str, "cash -5000\ngalhyp\n");
         }

         else
            str += sprintf(str, "j %s\n", Universe[next->pnum].name);
      }

      /* Save commands to file */

      if (len = strlen(&strbuf[0]))
      {
         if (fwrite(&strbuf[0], len, 1, fh) != 1)
         {
            printf("Error: Can't write to output file!\n");
            exit(0);
         }
      }
   }
}

/* Build a tree by recursively jumping to different planets and
 * buying/selling goods to maximize profit. After calling this
 * function, the global pointer 'BestSegmentNode' will be filled in
 * with the node pointer representing the end planet for a profitable
 * segment, or, if the required number of credits has been reached,
 * 'BestTerminatedNode' will be filled in (initialize it to NULL).
 * The sequence of jumps to get from the end node to the root node
 * can be recreated by backtracking the tree using the node's
 * parent field. This route represents a route SEGMENT in most
 * cases, since the tree won't be large enough to track the entire
 * route to get to the terminating credits. Therefore, to build
 * the entire route, you keep calling this function to get the
 * segments then concatenate the segments to form the final route.
 *
 * Some general points of interest:
 *
 * - We buy at each planet and sell at the next (there's no checking
 *   for extra profit to be made 'further down the line' by hanging
 *   on to goods). As a side-effect of this, it's possible that we
 *   can buy a particular trade good, jump to another system, sell
 *   the good, then buy it back! This is something which should be
 *   addressed.
 *
 * - The heuristics for termination could be improved. Currently,
 *   in the case of 'BestSegmentNode', we terminate on the leaf
 *   node which has the greatest number of credits, while termination
 *   for 'BestTerminatedNode' will select the node which has the
 *   lowest level (representing the fewest jumps) where the required
 *   number of credits was reached. This could be improved upon by
 *   looking ahead more.
 *
 * - The main concept used was the notion of 'gravitating' towards
 *   profitable regions of the galaxy. The deeper the tree, the
 *   more effective this technique should be.
 */
void scan_tree(struct Node *node)
{
   if (node->parent)
   {
      /* If prior to jumping here we equipped our ship with
       * a 35t capacity cargo bay then the originating state
       * buffer will flag this. This flag needs to be copied
       * to the node's flag mask now that we've arrived.
       * Similarly, if there was any cash left over after
       * buying goods we inherit it now.
       */
      StateBuffer = &node->parent->paths[node->pathnum].state;

      if (StateBuffer->fuel & 0x80)       /* Bit 7 flags "35t cargo bay" */
         node->flags |= NODEFLAG_LARGEBAY;

      if (node->parent->flags & NODEFLAG_LARGEBAY)
         node->flags |= NODEFLAG_LARGEBAY;
      
      node->cash = StateBuffer->cash;

      /* Sell any goods which we have in our cargo hold.
       *
       * The current algorithm sometimes sells goods and
       * then proceeds to buy some amount of the same good
       * back again once it figures out that buying the
       * good back again will result in profit at the
       * next planet. Obviously this could be improved
       * by hanging on to what we've already got. This
       * wasn't implemented for the Challenge, but it
       * would be easy enough to do.
       */
      ParentMarket = genmarket(get_market_fluctuation(node->mktseed, NULL), Universe[node->pnum]);

      for (Count = 0; Count <= lasttrade; Count++)
      {
         if (StateBuffer->goods[Count])
            node->cash += StateBuffer->goods[Count] * ParentMarket.price[Count];
      }

      /* Test for a route which gives us the most credits.
       * The heuristics for this are a bit simple-minded
       * and could do with some improvement. For example,
       * we should really be doing some lookahead work
       * here to test that we're not being led down a
       * bad path (eg: making money on our way to a low
       * profit region of the galaxy - we could be better
       * off following a less profitable route in the
       * short-term which takes us to a more profitable
       * region).
       */
      if (node->cash > MaxCash)
      {
         BestSegmentNode = node;
         MaxCash = node->cash;
      }

      if (node->cash >= Credits)
      {
         if (node->level < StopLevel)
         {
            StopLevel = node->level;
            BestTerminatedNode = node;
         }
      }

      /* Fill the fuel tank up. If we can't afford to do
       * this then we can start backtracking now (cash
       * can't become negative).
       */
      if ((node->cash -= (StateBuffer->fuel & 0x7f) * fuelcost) < 0)
         return;

      /* Check if we're a leaf node. If so, we're done with
       * this route and can start backtracking.
       */
      if (node->level == Tree.depth)
         return;
   }

   /* Scan for planets which are within jump range (we know that
    * we have a full tank of fuel so we have a 7.0LY jump radius).
    */
   for (;;)
   {
      if (node->visitcount == Tree.maxnodepaths)
         return;                 /* No more paths can be followed... */

      CurrentPlanet = node->pnum;
      PlanetStart = CurrentPlanet & 0xff00;     /* Start of this galaxy */
      PlanetStop = PlanetStart + 256;           /* End of this galaxy */
      FindCount = 0;                         /* # in-range planets found */

      for (; PlanetStart < PlanetStop; PlanetStart++)
      {
         Distance = get_distance(&Universe[PlanetStart], &Universe[CurrentPlanet]);

         if (Distance <= maxfuel)
         {
            if (PlanetStart != CurrentPlanet)
            {
               if (node->visitcount == FindCount++)
               {
                  PlanetStop = 0; /* Flag that we should go to PlanetStart */
                  break;
               }
            }
         }
      }     

      if (PlanetStop)
      {
         /* Check intergalactic path. We need to be at a system with
          * tech level 10 or higher for this since it requires the
          * purchase of an intergalactic hyperdrive. No fuel is used
          * for intergalactic jumps.
          *
          * Note: This route is unlikely to be profitable over the
          *       depth of the scan. It was only implemented out
          *       of interest.
          */
         if (node->cash >= INTERGALTHRESHOLD && Universe[CurrentPlanet].techlev >= 10)
         {
            node->cash -= INTERGALCOST;
            JumpNode = node->paths[node->visitcount].path;
            JumpNode->pathnum = node->visitcount++;
            JumpNode->pnum = (CurrentPlanet + 256) > 2047 ? CurrentPlanet & 0x00ff : CurrentPlanet + 256;
            buy_stuff(JumpNode);
            scan_tree(JumpNode);          /* Intergalactic jump */
         }

         return;                 /* No more planets to visit from here */
      }

      /* Check interstellar path (short range jump). Probe the
       * other system's market prices and compare with the current
       * market prices/availabilities to determine what we need
       * to buy here before making the jump. If we can't make any
       * profit from the jump, we go there empty (we go there anyway
       * because of the way the tree traversal works: we're looking
       * for profitable SEGMENTS, not profitable JUMPS).
       */
      node->paths[node->visitcount].state.fuel = Distance;
      JumpNode = node->paths[node->visitcount].path;
      JumpNode->pathnum = node->visitcount++;
      JumpNode->pnum = PlanetStart;       /* Stay in the same galaxy */
      buy_stuff(JumpNode);
      scan_tree(JumpNode);                /* Local jump */
   }
}

/* Buy goods/upgrade ship.
 *
 * curr is the system which we're jumping TO. Therefore the parent of
 * curr is the system which we're buying FROM.
 */
void buy_stuff(struct Node *curr)
{
   struct   State *state;
   ulong    *stopaddr;
   ulong    unitcost;
   ulong    units;
   ulong    cash;
   ulong    hold;
   ulong    i;
   ulong    j;    

   /* Build parent market in global storage */

   if (curr->parent->flags & NODEFLAG_LAVESTART)
   {
      /* Special case: Lave starting position */

      ParentMarket = genmarket(0x00,Universe[numforLave]);  /* Galaxy #1 */
      curr->mktseed = curr->parent->mktseed;
   }

   else
      ParentMarket = genmarket(get_market_fluctuation(curr->parent->mktseed, &curr->mktseed), Universe[curr->parent->pnum]);

   /* Build local market in global storage */

   LocalMarket = genmarket(get_market_fluctuation(curr->mktseed, NULL), Universe[curr->pnum]);

   /* Get cash from our parent */
   
   cash = curr->parent->cash;

   /* Get our total cargo hold space (which we know is empty)
    * and see if we can upgrade it to 35t capacity if it isn't
    * already.
    */
   state = &curr->parent->paths[curr->pathnum].state;

   hold = 20;                          /* Assume 20t cargo bay */

   if (curr->parent->flags & NODEFLAG_LARGEBAY)
      hold = 35;
      
   else
   {
      if (cash >= LARGEBAYTHRESHOLD)
      {
         state->fuel |= 0x80;       /* Flag so child node gets notified */
         cash -= LARGEBAYCOST;
         hold = 35;
      }
   }     

   /* Build an array of all goods which are available for purchase
    * and whose local prices are higher than the prices in the parent
    * market and of which we can afford to buy at least one unit.
    */
   for (i = 0, j = 0; i <= lasttrade; i++)
   {
      if (ParentMarket.quantity[i] && (LocalMarket.price[i] > ParentMarket.price[i])
         && (cash >= ParentMarket.price[i]))
      {
         /* Sort key format:
          *
          * bits 0-7   Commodity index (0 - 255).
          *
          * bits 8-31  Profit margin per unit.
          */          
         SortArray[j++] = ((LocalMarket.price[i] - ParentMarket.price[i]) << 8) | i;
      }
   }

   SortArray[j] = 0;                      /* Terminate array */
   stopaddr = &SortArray[j];

   /* Sort array into most profitable order (descending) */

   for (;;)
   {
      for (i = 0, j = 0; &SortArray[i] != stopaddr; i++)
      {
         if (SortArray[i] < SortArray[i+1])
         {
            j = SortArray[i];
            SortArray[i] = SortArray[i+1];
            SortArray[i+1] = j;
         }        
      }

      if (!j)
         break;

      stopaddr--;
   }

   if (SortArray[0])
   {
      /* Buying loop */

      for (i = 0; SortArray[i]; i++)
      {
         /* How much does each unit cost? */
         
         j = SortArray[i] & 0xff;
         unitcost = ParentMarket.price[j];

         if (units = (cash / unitcost))
         {
            /* We can afford to buy some of these. Now make
             * sure that we're not going to buy more than
             * what's available.
             */
            if (ParentMarket.quantity[j] < units)
               units = ParentMarket.quantity[j];

            if (commodities[j].units == tonnes)
            {
               /* Purchase will take up cargo space (tonnage) */
               
               if (hold >= units)
               {
                  /* Amount can be completely accommodated */

                  state->goods[j] = units;
                  cash -= units * unitcost;
                  hold -= units;

#if BUYDEBUG
                  printf("Buy  %ld%s of %s @ %.1f; Cash: %.1f Hold: %ld\n",
                        units, unitnames[commodities[j].units],
                        commodities[j].name, ((float)unitcost)/10.0,
                        ((float)cash)/10.0, hold);
#endif
               }

               else
               {
                  /* There's not enough cargo space to buy
                   * all that we can afford.
                   */
                  if (units = hold)
                  {
                     /* Buy as much as we can fit */

                     state->goods[j] = units;
                     cash -= units * unitcost;
                     hold = 0;

#if BUYDEBUG
                  printf("Buy  %ld%s of %s @ %.1f; Cash: %.1f Hold: %ld\n",
                        units, unitnames[commodities[j].units],
                        commodities[j].name, ((float)unitcost)/10.0,
                        ((float)cash)/10.0, hold);
#endif
                  }
               }
            }

            else
            {
               /* This purchase won't take up any space so buy
                * all that we can afford.
                */
               state->goods[j] = units;
               cash -= units * unitcost;

#if BUYDEBUG
                  printf("Buy  %ld%s of %s @ %.1f; Cash: %.1f Hold: %ld\n",
                        units, unitnames[commodities[j].units],
                        commodities[j].name, ((float)unitcost)/10.0,
                        ((float)cash)/10.0, hold);
#endif
            }
         }
      }
   }  

   else
   {
      /* There's nothing we can buy which we can profit from!
       * We'll jump to the next system without buying from this
       * system rather than lose money by buying here (yes,
       * this case can occur in Elite).
       */
   }

   /* Note any left-over cash. This gets passed on to our child
    * node after we make the jump to the next system.
    */
   state->cash = cash;
}

/* Return distance between two planets in the same galaxy.
 */
ulong get_distance(plansys *a,plansys *b)
/* Seperation between two planets (4*sqrt(X*X+Y*Y/4)) */
{  return (ulong)ftoi(4*sqrt((a->x-b->x)*(a->x-b->x)+(a->y-b->y)*(a->y-b->y)/4));
}

/* Return market fluctuation value given a market seed and store
 * aside the new seed if newseed != NULL.
 */
uint get_market_fluctuation(int seed, int *newseed)
{
   uint  fluc;

   lastrand = seed;
   fluc = port_rand();

   if (newseed)
      *newseed = lastrand;

   return (uint)(fluc & 0xff);
}

/* Precalculate Elite universe.
 */
void create_universe(void)
{
   ulong    totalcount = 0;

   ulong    galcount;
   ulong    syscount;
   ulong    c;

   for (c = 1; c < 9; c++)
   {
      /* Initialize seed for galaxy #1 */
      seed.w0=base0; seed.w1=base1; seed.w2=base2;

      /* Generate galaxy #c */
      for(galcount=1;galcount<c;++galcount)
         nextgalaxy(&seed);

      for(syscount=0;syscount<galsize;++syscount)
         Universe[totalcount++]=makesystem(&seed);
   }
}

/********************** TXTELITE CODE BY IAN BELL *************************/

/**- General functions **/

uint mymin(uint a,uint b) { if(a<b) return(a);  else return(b);}

void stop(char * string)
{ printf("\n%s",string);
  exit(1);
}

 /**+  ftoi **/
signed int ftoi(double value)
{ return ((signed int)floor(value+0.5));
}

 /**+  ftoi2 **/
signed int ftoi2(double value)
{ return ((signed int)floor(value));
}

void stripout(char *s,const char c) /* Remove all c's from string s */
{   size_t i=0,j=0;
    while(i<strlen(s))
    { if(s[i]!=c) { s[j]=s[i]; j++;}
      i++;
    }
    s[j]=0;
}

void tweakseed(seedtype *s)
{ uint16 temp;
  temp = ((*s).w0)+((*s).w1)+((*s).w2); /* 2 byte aritmetic */
  (*s).w0 = (*s).w1;
  (*s).w1 = (*s).w2;
  (*s).w2 = temp;
}

/**-Functions for stock market **/

markettype genmarket(uint fluct, plansys p)
/* Prices and availabilities are influenced by the planet's economy type
   (0-7) and a random "fluctuation" byte that was kept within the saved
   commander position to keep the market prices constant over gamesaves.
   Availabilities must be saved with the game since the player alters them
   by buying (and selling(?))

   Almost all operations are one byte only and overflow "errors" are
   extremely frequent and exploited.

   Trade Item prices are held internally in a single byte=true value/4.
   The decimal point in prices is introduced only when printing them.
   Internally, all prices are integers.
   The player's cash is held in four bytes. 
 */

{  markettype market;
  unsigned short i;
  for(i=0;i<=lasttrade;i++)
  {   signed int q; 
    signed int product = (p.economy)*(commodities[i].gradient);
    signed int changing = fluct & (commodities[i].maskbyte);
      q =  (commodities[i].basequant) + changing - product; 
    q = q&0xFF;
    if(q&0x80) {q=0;};                       /* Clip to positive 8-bit */

    market.quantity[i] = (uint16)(q & 0x3F); /* Mask to 6 bits */

    q =  (commodities[i].baseprice) + changing + product;
    q = q & 0xFF;
    market.price[i] = (uint16) (q*4);
  }
   market.quantity[AlienItems] = 0; /* Override to force nonavailability */
   return market;
}

/**-Generate system info from seed **/

plansys makesystem(seedtype *s)
{  plansys thissys;
  uint pair1,pair2,pair3,pair4;
  uint16 longnameflag=((*s).w0)&64;
 
  thissys.x=(((*s).w1)>>8);
  thissys.y=(((*s).w0)>>8);

  thissys.govtype =((((*s).w1)>>3)&7); /* bits 3,4 &5 of w1 */

  thissys.economy =((((*s).w0)>>8)&7); /* bits 8,9 &A of w0 */
  if (thissys.govtype <=1)
  { thissys.economy = ((thissys.economy)|2);
  } 

  thissys.techlev =((((*s).w1)>>8)&3)+((thissys.economy)^7);
  thissys.techlev +=((thissys.govtype)>>1);
  if (((thissys.govtype)&1)==1)  thissys.techlev+=1;
   /* C simulation of 6502's LSR then ADC */
 
  thissys.population = 4*(thissys.techlev) + (thissys.economy);
  thissys.population +=  (thissys.govtype) + 1;

  thissys.productivity = (((thissys.economy)^7)+3)*((thissys.govtype)+4);
  thissys.productivity *= (thissys.population)*8;

  thissys.radius = 256*(((((*s).w2)>>8)&15)+11) + thissys.x;  

   thissys.goatsoupseed.a = (*s).w1 & 0xFF;;
   thissys.goatsoupseed.b = (*s).w1 >>8;
   thissys.goatsoupseed.c = (*s).w2 & 0xFF;
   thissys.goatsoupseed.d = (*s).w2 >> 8;

  pair1=2*((((*s).w2)>>8)&31);  tweakseed(s);
  pair2=2*((((*s).w2)>>8)&31);  tweakseed(s);
  pair3=2*((((*s).w2)>>8)&31);  tweakseed(s);
  pair4=2*((((*s).w2)>>8)&31);   tweakseed(s);
   /* Always four iterations of random number */

  (thissys.name)[0]=pairs[pair1];
  (thissys.name)[1]=pairs[pair1+1];
  (thissys.name)[2]=pairs[pair2];
  (thissys.name)[3]=pairs[pair2+1];
  (thissys.name)[4]=pairs[pair3];
  (thissys.name)[5]=pairs[pair3+1];

  if(longnameflag) /* bit 6 of ORIGINAL w0 flags a four-pair name */
  {
  (thissys.name)[6]=pairs[pair4];
  (thissys.name)[7]=pairs[pair4+1];
  (thissys.name)[8]=0;
  }
  else (thissys.name)[6]=0;
  stripout(thissys.name,'.');

return thissys;
}  

/**+Generate galaxy **/

/* Functions for galactic hyperspace */

uint16 rotatel(uint16 x) /* rotate 8 bit number leftwards */
  /* (tried to use chars but too much effort persuading this braindead
     language to do bit operations on bytes!) */
{ uint16 temp = x&128;
   return (2*(x&127))+(temp>>7);
} 

uint16 twist(uint16 x)
{ return (uint16)((256*rotatel(x>>8))+rotatel(x&255));
} 

void nextgalaxy(seedtype *s) /* Apply to base seed; once for galaxy 2  */
{ (*s).w0 = twist((*s).w0);  /* twice for galaxy 3, etc. */
  (*s).w1 = twist((*s).w1);  /* Eighth application gives galaxy 1 again*/
  (*s).w2 = twist((*s).w2);
}

/* END */
